# Copyright (C) The Arvados Authors. All rights reserved.
#
# SPDX-License-Identifier: Apache-2.0

#' Arvados
#'
#' This class implements a full REST client to the Arvados API.
#'
#' @examples
#' \dontrun{
#' arv <- Arvados$new("your Arvados token", "example.arvadosapi.com")
#'
#' collection <- arv$collections.get("uuid")
#'
#' collectionList <- arv$collections.list(list(list("name", "like", "Test%")))
#' collectionList <- listAll(arv$collections.list, list(list("name", "like", "Test%")))
#'
#' deletedCollection <- arv$collections.delete("uuid")
#'
#' updatedCollection <- arv$collections.update(list(name = "New name", description = "New description"),
#'                                             "uuid")
#'
#' createdCollection <- arv$collections.create(list(name = "Example",
#'                                                  description = "This is a test collection"))
#' }

#' @export
Arvados <- R6::R6Class(

	"Arvados",

	public = list(

		#' @description Create a new Arvados API client.
		#' @param authToken Authentification token. If not specified ARVADOS_API_TOKEN environment variable will be used.
		#' @param hostName Host name. If not specified ARVADOS_API_HOST environment variable will be used.
		#' @param numRetries Number which specifies how many times to retry failed service requests.
		#' @return A new `Arvados` object.
		initialize = function(authToken = NULL, hostName = NULL, numRetries = 0)
		{
			if(!is.null(hostName))
				Sys.setenv(ARVADOS_API_HOST = hostName)

			if(!is.null(authToken))
				Sys.setenv(ARVADOS_API_TOKEN = authToken)

			hostName <- Sys.getenv("ARVADOS_API_HOST")
			token    <- Sys.getenv("ARVADOS_API_TOKEN")

			if(hostName == "" | token == "")
				stop(paste("Please provide host name and authentification token",
						   "or set ARVADOS_API_HOST and ARVADOS_API_TOKEN",
						   "environment variables."))

			private$token <- token
			private$host  <- paste0("https://", hostName, "/arvados/v1/")
			private$numRetries <- numRetries
			private$REST <- RESTService$new(token, hostName,
			                                HttpRequest$new(), HttpParser$new(),
			                                numRetries)

		},

		#' @description Get a ApiClientAuthorization record by UUID.
		#' @param uuid The UUID of the ApiClientAuthorization to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return ApiClientAuthorization object.
		api_client_authorizations_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("api_client_authorizations/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a ApiClientAuthorizationList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return ApiClientAuthorizationList object.
		api_client_authorizations_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("api_client_authorizations")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new ApiClientAuthorization.
		#' @param apiClientAuthorization ApiClientAuthorization object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return ApiClientAuthorization object.
		api_client_authorizations_create = function(apiClientAuthorization,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("api_client_authorizations")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(apiClientAuthorization) > 0)
				body <- jsonlite::toJSON(list(apiClientAuthorization = apiClientAuthorization), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing ApiClientAuthorization.
		#' @param apiClientAuthorization ApiClientAuthorization object.
		#' @param uuid The UUID of the ApiClientAuthorization to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return ApiClientAuthorization object.
		api_client_authorizations_update = function(apiClientAuthorization,
			uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("api_client_authorizations/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(apiClientAuthorization) > 0)
				body <- jsonlite::toJSON(list(apiClientAuthorization = apiClientAuthorization), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing ApiClientAuthorization.
		#' @param uuid The UUID of the ApiClientAuthorization to delete.
		#' @return ApiClientAuthorization object.
		api_client_authorizations_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("api_client_authorizations/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a token for the system ("root") user.
		#' @param scopes An array of strings defining the scope of resources this token will be allowed to access. Refer to the [scopes reference][] for details.
		#' 
		#' [scopes reference]: https://doc.arvados.org/api/tokens.html#scopes
		#' 
		#' @return ApiClientAuthorization object.
		api_client_authorizations_create_system_auth = function(scopes = NULL)
		{
			endPoint <- stringr::str_interp("api_client_authorizations/create_system_auth")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(scopes = scopes)
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return all metadata for the token used to authorize this request.
		#' @return ApiClientAuthorization object.
		api_client_authorizations_current = function()
		{
			endPoint <- stringr::str_interp("api_client_authorizations/current")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a AuthorizedKey record by UUID.
		#' @param uuid The UUID of the AuthorizedKey to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return AuthorizedKey object.
		authorized_keys_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("authorized_keys/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a AuthorizedKeyList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return AuthorizedKeyList object.
		authorized_keys_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("authorized_keys")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new AuthorizedKey.
		#' @param authorizedKey AuthorizedKey object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return AuthorizedKey object.
		authorized_keys_create = function(authorizedKey,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("authorized_keys")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(authorizedKey) > 0)
				body <- jsonlite::toJSON(list(authorizedKey = authorizedKey), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing AuthorizedKey.
		#' @param authorizedKey AuthorizedKey object.
		#' @param uuid The UUID of the AuthorizedKey to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return AuthorizedKey object.
		authorized_keys_update = function(authorizedKey, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("authorized_keys/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(authorizedKey) > 0)
				body <- jsonlite::toJSON(list(authorizedKey = authorizedKey), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing AuthorizedKey.
		#' @param uuid The UUID of the AuthorizedKey to delete.
		#' @return AuthorizedKey object.
		authorized_keys_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("authorized_keys/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Collection record by UUID.
		#' @param uuid The UUID of the Collection to return.
		#' @param select An array of names of attributes to return in the response.
		#' @param includeTrash Show collection even if its `is_trashed` attribute is true.
		#' @return Collection object.
		collections_get = function(uuid, select = NULL, includeTrash = NULL)
		{
			endPoint <- stringr::str_interp("collections/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, includeTrash = includeTrash)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a CollectionList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include collections whose `is_trashed` attribute is true.
		#' @param includeOldVersions Include past collection versions.
		#' @return CollectionList object.
		collections_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL, includeOldVersions = NULL)
		{
			endPoint <- stringr::str_interp("collections")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation,
							  includeTrash = includeTrash, includeOldVersions = includeOldVersions)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Collection.
		#' @param collection Collection object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @param replaceFiles Add, delete, and replace files and directories with new content
		#' and/or content from other collections. Refer to the
		#' [replace_files reference][] for details.
		#' 
		#' [replace_files reference]: https://doc.arvados.org/api/methods/collections.html#replace_files
		#' 
		#' 
		#' @param replaceSegments Replace existing block segments in the collection with new segments.
		#' Refer to the [replace_segments reference][] for details.
		#' 
		#' [replace_segments reference]: https://doc.arvados.org/api/methods/collections.html#replace_segments
		#' 
		#' 
		#' @return Collection object.
		collections_create = function(collection,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL,
			replaceFiles = NULL, replaceSegments = NULL)
		{
			endPoint <- stringr::str_interp("collections")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID, replaceFiles = replaceFiles,
							  replaceSegments = replaceSegments)
			
			if(length(collection) > 0)
				body <- jsonlite::toJSON(list(collection = collection), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Collection.
		#' @param collection Collection object.
		#' @param uuid The UUID of the Collection to update.
		#' @param select An array of names of attributes to return in the response.
		#' @param replaceFiles Add, delete, and replace files and directories with new content
		#' and/or content from other collections. Refer to the
		#' [replace_files reference][] for details.
		#' 
		#' [replace_files reference]: https://doc.arvados.org/api/methods/collections.html#replace_files
		#' 
		#' 
		#' @param replaceSegments Replace existing block segments in the collection with new segments.
		#' Refer to the [replace_segments reference][] for details.
		#' 
		#' [replace_segments reference]: https://doc.arvados.org/api/methods/collections.html#replace_segments
		#' 
		#' 
		#' @return Collection object.
		collections_update = function(collection,
			uuid, select = NULL, replaceFiles = NULL,
			replaceSegments = NULL)
		{
			endPoint <- stringr::str_interp("collections/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, replaceFiles = replaceFiles,
							  replaceSegments = replaceSegments)
			
			if(length(collection) > 0)
				body <- jsonlite::toJSON(list(collection = collection), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Collection.
		#' @param uuid The UUID of the Collection to delete.
		#' @return Collection object.
		collections_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("collections/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Detail the provenance of a given collection.
		#' @param uuid The UUID of the Collection to query.
		#' @return Collection object.
		collections_provenance = function(uuid)
		{
			endPoint <- stringr::str_interp("collections/${uuid}/provenance")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Detail where a given collection has been used.
		#' @param uuid The UUID of the Collection to query.
		#' @return Collection object.
		collections_used_by = function(uuid)
		{
			endPoint <- stringr::str_interp("collections/${uuid}/used_by")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Trash a collection.
		#' @param uuid The UUID of the Collection to update.
		#' @return Collection object.
		collections_trash = function(uuid)
		{
			endPoint <- stringr::str_interp("collections/${uuid}/trash")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Untrash a collection.
		#' @param uuid The UUID of the Collection to update.
		#' @return Collection object.
		collections_untrash = function(uuid)
		{
			endPoint <- stringr::str_interp("collections/${uuid}/untrash")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a ComputedPermissionList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @return ComputedPermissionList object.
		computed_permissions_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, count = NULL)
		{
			endPoint <- stringr::str_interp("computed_permissions")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, count = count)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Container record by UUID.
		#' @param uuid The UUID of the Container to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return Container object.
		containers_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("containers/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a ContainerList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return ContainerList object.
		containers_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("containers")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Container.
		#' @param container Container object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return Container object.
		containers_create = function(container, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("containers")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(container) > 0)
				body <- jsonlite::toJSON(list(container = container), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Container.
		#' @param container Container object.
		#' @param uuid The UUID of the Container to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return Container object.
		containers_update = function(container, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("containers/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(container) > 0)
				body <- jsonlite::toJSON(list(container = container), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Container.
		#' @param uuid The UUID of the Container to delete.
		#' @return Container object.
		containers_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get the API client authorization token associated with this container.
		#' @param uuid The UUID of the Container to query.
		#' @return Container object.
		containers_auth = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}/auth")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Lock a container (for a dispatcher to begin running it).
		#' @param uuid The UUID of the Container to update.
		#' @return Container object.
		containers_lock = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}/lock")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Unlock a container (for a dispatcher to stop running it).
		#' @param uuid The UUID of the Container to update.
		#' @return Container object.
		containers_unlock = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}/unlock")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Recalculate and return the priority of a given container.
		#' @param uuid The UUID of the Container to update.
		#' @return Container object.
		containers_update_priority = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}/update_priority")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return secret mount information for the container associated with the API token authorizing this request.
		#' @param uuid The UUID of the Container to query.
		#' @return Container object.
		containers_secret_mounts = function(uuid)
		{
			endPoint <- stringr::str_interp("containers/${uuid}/secret_mounts")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return the container record associated with the API token authorizing this request.
		#' @return Container object.
		containers_current = function()
		{
			endPoint <- stringr::str_interp("containers/current")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a ContainerRequest record by UUID.
		#' @param uuid The UUID of the ContainerRequest to return.
		#' @param select An array of names of attributes to return in the response.
		#' @param includeTrash Show container request even if its owner project is trashed.
		#' @return ContainerRequest object.
		container_requests_get = function(uuid, select = NULL, includeTrash = NULL)
		{
			endPoint <- stringr::str_interp("container_requests/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, includeTrash = includeTrash)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a ContainerRequestList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include container requests whose owner project is trashed.
		#' @return ContainerRequestList object.
		container_requests_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL)
		{
			endPoint <- stringr::str_interp("container_requests")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation,
							  includeTrash = includeTrash)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new ContainerRequest.
		#' @param containerRequest ContainerRequest object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return ContainerRequest object.
		container_requests_create = function(containerRequest,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("container_requests")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(containerRequest) > 0)
				body <- jsonlite::toJSON(list(containerRequest = containerRequest), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing ContainerRequest.
		#' @param containerRequest ContainerRequest object.
		#' @param uuid The UUID of the ContainerRequest to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return ContainerRequest object.
		container_requests_update = function(containerRequest, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("container_requests/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(containerRequest) > 0)
				body <- jsonlite::toJSON(list(containerRequest = containerRequest), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing ContainerRequest.
		#' @param uuid The UUID of the ContainerRequest to delete.
		#' @return ContainerRequest object.
		container_requests_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("container_requests/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return scheduling details for a container request.
		#' @param uuid The UUID of the container request to query.
		#' @return ContainerRequest object.
		container_requests_container_status = function(uuid)
		{
			endPoint <- stringr::str_interp("container_requests/${uuid}/container_status")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(uuid = uuid)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Group record by UUID.
		#' @param uuid The UUID of the Group to return.
		#' @param select An array of names of attributes to return in the response.
		#' @param includeTrash Return group/project even if its `is_trashed` attribute is true.
		#' @return Group object.
		groups_get = function(uuid, select = NULL, includeTrash = NULL)
		{
			endPoint <- stringr::str_interp("groups/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, includeTrash = includeTrash)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a GroupList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include items whose `is_trashed` attribute is true.
		#' @return GroupList object.
		groups_list = function(filters = NULL, where = NULL,
			order = NULL, select = NULL, distinct = NULL,
			limit = NULL, offset = NULL, count = NULL,
			clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL)
		{
			endPoint <- stringr::str_interp("groups")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation,
							  includeTrash = includeTrash)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Group.
		#' @param group Group object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @param async If true, cluster permission will not be updated immediately, but instead at the next configured update interval.
		#' @return Group object.
		groups_create = function(group, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL,
			async = NULL)
		{
			endPoint <- stringr::str_interp("groups")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID, async = async)
			
			if(length(group) > 0)
				body <- jsonlite::toJSON(list(group = group), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Group.
		#' @param group Group object.
		#' @param uuid The UUID of the Group to update.
		#' @param select An array of names of attributes to return in the response.
		#' @param async If true, cluster permission will not be updated immediately, but instead at the next configured update interval.
		#' @return Group object.
		groups_update = function(group, uuid, select = NULL, async = NULL)
		{
			endPoint <- stringr::str_interp("groups/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, async = async)
			
			if(length(group) > 0)
				body <- jsonlite::toJSON(list(group = group), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Group.
		#' @param uuid The UUID of the Group to delete.
		#' @return Group object.
		groups_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("groups/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List objects that belong to a group.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include items whose `is_trashed` attribute is true.
		#' @param uuid If given, limit the listing to objects owned by the
		#' user or group with this UUID.
		#' @param recursive If true, include contents from child groups recursively.
		#' @param include An array of referenced objects to include in the `included` field of the response. Supported values in the array are:
		#' 
		#'   * `"container_uuid"`
		#'   * `"owner_uuid"`
		#' 
		#' 
		#' @param includeOldVersions If true, include past versions of collections in the listing.
		#' @param excludeHomeProject If true, exclude contents of the user's home project from the listing.
		#' Calling this method with this flag set is how clients enumerate objects shared
		#' with the current user.
		#' @return Group object.
		groups_contents = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL, uuid = NULL, recursive = NULL,
			include = NULL, includeOldVersions = NULL,
			excludeHomeProject = NULL)
		{
			endPoint <- stringr::str_interp("groups/contents")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation,
							  includeTrash = includeTrash, uuid = uuid,
							  recursive = recursive, include = include,
							  includeOldVersions = includeOldVersions,
							  excludeHomeProject = excludeHomeProject)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List groups that the current user can access via permission links.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include items whose `is_trashed` attribute is true.
		#' @param include A string naming referenced objects to include in the `included` field of the response. Supported values are:
		#' 
		#'   * `"owner_uuid"`
		#' 
		#' 
		#' @return Group object.
		groups_shared = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL, include = NULL)
		{
			endPoint <- stringr::str_interp("groups/shared")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation,
							  includeTrash = includeTrash, include = include)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Trash a group.
		#' @param uuid The UUID of the Group to update.
		#' @return Group object.
		groups_trash = function(uuid)
		{
			endPoint <- stringr::str_interp("groups/${uuid}/trash")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Untrash a group.
		#' @param uuid The UUID of the Group to update.
		#' @return Group object.
		groups_untrash = function(uuid)
		{
			endPoint <- stringr::str_interp("groups/${uuid}/untrash")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a KeepService record by UUID.
		#' @param uuid The UUID of the KeepService to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return KeepService object.
		keep_services_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("keep_services/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a KeepServiceList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return KeepServiceList object.
		keep_services_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("keep_services")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new KeepService.
		#' @param keepService KeepService object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return KeepService object.
		keep_services_create = function(keepService,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("keep_services")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(keepService) > 0)
				body <- jsonlite::toJSON(list(keepService = keepService), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing KeepService.
		#' @param keepService KeepService object.
		#' @param uuid The UUID of the KeepService to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return KeepService object.
		keep_services_update = function(keepService, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("keep_services/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(keepService) > 0)
				body <- jsonlite::toJSON(list(keepService = keepService), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing KeepService.
		#' @param uuid The UUID of the KeepService to delete.
		#' @return KeepService object.
		keep_services_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("keep_services/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List Keep services that the current client can access.
		#' @return KeepService object.
		keep_services_accessible = function()
		{
			endPoint <- stringr::str_interp("keep_services/accessible")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Link record by UUID.
		#' @param uuid The UUID of the Link to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return Link object.
		links_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("links/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a LinkList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return LinkList object.
		links_list = function(filters = NULL, where = NULL,
			order = NULL, select = NULL, distinct = NULL,
			limit = NULL, offset = NULL, count = NULL,
			clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("links")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Link.
		#' @param link Link object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return Link object.
		links_create = function(link, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("links")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(link) > 0)
				body <- jsonlite::toJSON(list(link = link), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Link.
		#' @param link Link object.
		#' @param uuid The UUID of the Link to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return Link object.
		links_update = function(link, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("links/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(link) > 0)
				body <- jsonlite::toJSON(list(link = link), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Link.
		#' @param uuid The UUID of the Link to delete.
		#' @return Link object.
		links_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("links/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List permissions granted on an Arvados object.
		#' @param uuid The UUID of the Link to query.
		#' @return Link object.
		links_get_permissions = function(uuid)
		{
			endPoint <- stringr::str_interp("permissions/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Log record by UUID.
		#' @param uuid The UUID of the Log to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return Log object.
		logs_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("logs/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a LogList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return LogList object.
		logs_list = function(filters = NULL, where = NULL,
			order = NULL, select = NULL, distinct = NULL,
			limit = NULL, offset = NULL, count = NULL,
			clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("logs")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Log.
		#' @param log Log object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return Log object.
		logs_create = function(log, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("logs")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(log) > 0)
				body <- jsonlite::toJSON(list(log = log), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Log.
		#' @param log Log object.
		#' @param uuid The UUID of the Log to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return Log object.
		logs_update = function(log, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("logs/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(log) > 0)
				body <- jsonlite::toJSON(list(log = log), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Log.
		#' @param uuid The UUID of the Log to delete.
		#' @return Log object.
		logs_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("logs/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a User record by UUID.
		#' @param uuid The UUID of the User to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return User object.
		users_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("users/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a UserList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return UserList object.
		users_list = function(filters = NULL, where = NULL,
			order = NULL, select = NULL, distinct = NULL,
			limit = NULL, offset = NULL, count = NULL,
			clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("users")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new User.
		#' @param user User object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return User object.
		users_create = function(user, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("users")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(user) > 0)
				body <- jsonlite::toJSON(list(user = user), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing User.
		#' @param user User object.
		#' @param uuid The UUID of the User to update.
		#' @param select An array of names of attributes to return in the response.
		#' @param bypassFederation If true, do not try to update the user on any other clusters in the federation,
		#' only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' @return User object.
		users_update = function(user, uuid, select = NULL,
			bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("users/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, bypassFederation = bypassFederation)
			
			if(length(user) > 0)
				body <- jsonlite::toJSON(list(user = user), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing User.
		#' @param uuid The UUID of the User to delete.
		#' @return User object.
		users_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("users/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return the user record associated with the API token authorizing this request.
		#' @return User object.
		users_current = function()
		{
			endPoint <- stringr::str_interp("users/current")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Return this cluster's system ("root") user record.
		#' @return User object.
		users_system = function()
		{
			endPoint <- stringr::str_interp("users/system")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Set the `is_active` flag on a user record.
		#' @param uuid The UUID of the User to update.
		#' @return User object.
		users_activate = function(uuid)
		{
			endPoint <- stringr::str_interp("users/${uuid}/activate")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Convenience method to "fully" set up a user record with a virtual machine login and notification email.
		#' @param uuid UUID of an existing user record to set up.
		#' @param user Attributes of a new user record to set up.
		#' @param repoName This parameter is obsolete and ignored.
		#' @param vmUUID If given, setup creates a login link to allow this user to access the Arvados virtual machine with this UUID.
		#' @param sendNotificationEmail If true, send an email to the user notifying them they can now access this Arvados cluster.
		#' @return User object.
		users_setup = function(uuid = NULL, user = NULL,
			repoName = NULL, vmUUID = NULL, sendNotificationEmail = NULL)
		{
			endPoint <- stringr::str_interp("users/setup")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(uuid = uuid, user = user,
							  repoName = repoName, vmUUID = vmUUID, sendNotificationEmail = sendNotificationEmail)
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Unset a user's active flag and delete associated records.
		#' @param uuid The UUID of the User to update.
		#' @return User object.
		users_unsetup = function(uuid)
		{
			endPoint <- stringr::str_interp("users/${uuid}/unsetup")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Transfer ownership of one user's data to another.
		#' @param newOwnerUUID UUID of the user or group that will take ownership of data owned by the old user.
		#' @param newUserToken Valid API token for the user receiving ownership. If you use this option, it takes ownership of data owned by the user making the request.
		#' @param redirectToNewUser If true, authorization attempts for the old user will be redirected to the new user.
		#' @param oldUserUUID UUID of the user whose ownership is being transferred to `new_owner_uuid`. You must be an admin to use this option.
		#' @param newUserUUID UUID of the user receiving ownership. You must be an admin to use this option.
		#' @return User object.
		users_merge = function(newOwnerUUID, newUserToken = NULL,
			redirectToNewUser = NULL, oldUserUUID = NULL,
			newUserUUID = NULL)
		{
			endPoint <- stringr::str_interp("users/merge")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(newOwnerUUID = newOwnerUUID,
							  newUserToken = newUserToken, redirectToNewUser = redirectToNewUser,
							  oldUserUUID = oldUserUUID, newUserUUID = newUserUUID)
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a UserAgreement record by UUID.
		#' @param uuid The UUID of the UserAgreement to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return UserAgreement object.
		user_agreements_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("user_agreements/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a UserAgreementList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return UserAgreementList object.
		user_agreements_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("user_agreements")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new UserAgreement.
		#' @param userAgreement UserAgreement object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return UserAgreement object.
		user_agreements_create = function(userAgreement,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("user_agreements")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(userAgreement) > 0)
				body <- jsonlite::toJSON(list(userAgreement = userAgreement), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing UserAgreement.
		#' @param userAgreement UserAgreement object.
		#' @param uuid The UUID of the UserAgreement to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return UserAgreement object.
		user_agreements_update = function(userAgreement, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("user_agreements/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(userAgreement) > 0)
				body <- jsonlite::toJSON(list(userAgreement = userAgreement), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing UserAgreement.
		#' @param uuid The UUID of the UserAgreement to delete.
		#' @return UserAgreement object.
		user_agreements_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("user_agreements/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List all user agreement signature links from a user.
		#' @return UserAgreement object.
		user_agreements_signatures = function()
		{
			endPoint <- stringr::str_interp("user_agreements/signatures")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a signature link from the current user for a given user agreement.
		#' @return UserAgreement object.
		user_agreements_sign = function()
		{
			endPoint <- stringr::str_interp("user_agreements/sign")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a VirtualMachine record by UUID.
		#' @param uuid The UUID of the VirtualMachine to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return VirtualMachine object.
		virtual_machines_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("virtual_machines/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a VirtualMachineList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return VirtualMachineList object.
		virtual_machines_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("virtual_machines")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new VirtualMachine.
		#' @param virtualMachine VirtualMachine object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return VirtualMachine object.
		virtual_machines_create = function(virtualMachine,
			select = NULL, ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("virtual_machines")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(virtualMachine) > 0)
				body <- jsonlite::toJSON(list(virtualMachine = virtualMachine), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing VirtualMachine.
		#' @param virtualMachine VirtualMachine object.
		#' @param uuid The UUID of the VirtualMachine to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return VirtualMachine object.
		virtual_machines_update = function(virtualMachine, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("virtual_machines/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(virtualMachine) > 0)
				body <- jsonlite::toJSON(list(virtualMachine = virtualMachine), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing VirtualMachine.
		#' @param uuid The UUID of the VirtualMachine to delete.
		#' @return VirtualMachine object.
		virtual_machines_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("virtual_machines/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List login permission links for a given virtual machine.
		#' @param uuid The UUID of the VirtualMachine to query.
		#' @return VirtualMachine object.
		virtual_machines_logins = function(uuid)
		{
			endPoint <- stringr::str_interp("virtual_machines/${uuid}/logins")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description List login permission links for all virtual machines.
		#' @return VirtualMachine object.
		virtual_machines_get_all_logins = function()
		{
			endPoint <- stringr::str_interp("virtual_machines/get_all_logins")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get a Workflow record by UUID.
		#' @param uuid The UUID of the Workflow to return.
		#' @param select An array of names of attributes to return in the response.
		#' @return Workflow object.
		workflows_get = function(uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("workflows/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Retrieve a WorkflowList.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @return WorkflowList object.
		workflows_list = function(filters = NULL,
			where = NULL, order = NULL, select = NULL,
			distinct = NULL, limit = NULL, offset = NULL,
			count = NULL, clusterID = NULL, bypassFederation = NULL)
		{
			endPoint <- stringr::str_interp("workflows")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(filters = filters, where = where,
							  order = order, select = select, distinct = distinct,
							  limit = limit, offset = offset, count = count,
							  clusterID = clusterID, bypassFederation = bypassFederation)
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Create a new Workflow.
		#' @param workflow Workflow object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @return Workflow object.
		workflows_create = function(workflow, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL)
		{
			endPoint <- stringr::str_interp("workflows")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select, ensureUniqueName = ensureUniqueName,
							  clusterID = clusterID)
			
			if(length(workflow) > 0)
				body <- jsonlite::toJSON(list(workflow = workflow), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				if (identical(sub('Entity:.*', '', resource$errors), '//railsapi.internal/arvados/v1/collections: 422 Unprocessable ')) {
					resource <- cat(format('An object with the given name already exists with this owner. If you want to update it use the update method instead'))
				} else {
					stop(resource$errors)
				}
			}
			
			resource
		},

		#' @description Update attributes of an existing Workflow.
		#' @param workflow Workflow object.
		#' @param uuid The UUID of the Workflow to update.
		#' @param select An array of names of attributes to return in the response.
		#' @return Workflow object.
		workflows_update = function(workflow, uuid, select = NULL)
		{
			endPoint <- stringr::str_interp("workflows/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- list(select = select)
			
			if(length(workflow) > 0)
				body <- jsonlite::toJSON(list(workflow = workflow), 
				                         auto_unbox = TRUE)
			else
				body <- NULL
			
			response <- private$REST$http$exec("PUT", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Delete an existing Workflow.
		#' @param uuid The UUID of the Workflow to delete.
		#' @return Workflow object.
		workflows_delete = function(uuid)
		{
			endPoint <- stringr::str_interp("workflows/${uuid}")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("DELETE", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get this cluster's public configuration settings.
		#' @return  object.
		configs_get = function()
		{
			endPoint <- stringr::str_interp("config")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Get this cluster's configured vocabulary definition.
		#' 
		#' Refer to [metadata vocabulary documentation][] for details.
		#' 
		#' [metadata vocabulary documentation]: https://doc.aravdos.org/admin/metadata-vocabulary.html
		#' 
		#' 
		#' @return  object.
		vocabularies_get = function()
		{
			endPoint <- stringr::str_interp("vocabulary")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("GET", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description Run scheduled data trash and sweep operations across this cluster's Keep services.
		#' @return  object.
		sys_get = function()
		{
			endPoint <- stringr::str_interp("sys/trash_sweep")
			url <- paste0(private$host, endPoint)
			headers <- list(Authorization = paste("Bearer", private$token), 
			                "Content-Type" = "application/json")
			queryArgs <- NULL
			
			body <- NULL
			
			response <- private$REST$http$exec("POST", url, headers, body,
			                                   queryArgs, private$numRetries)
			resource <- private$REST$httpParser$parseJSONResponse(response)
			
			if(!is.null(resource$errors)) {
				stop(resource$errors)
			}
			
			resource
		},

		#' @description An alias for `groups_get`.
		#' @param uuid The UUID of the Group to return.
		#' @param select An array of names of attributes to return in the response.
		#' @param includeTrash Return group/project even if its `is_trashed` attribute is true.
		#' @return A Group object.
		project_get = function(uuid, select = NULL, includeTrash = NULL)
		{
			self$groups_get(uuid = uuid, select = select, includeTrash = includeTrash)
		},

		#' @description A wrapper for `groups_create` that sets `group_class="project"`.
		#' @param group Group object.
		#' @param select An array of names of attributes to return in the response.
		#' @param ensureUniqueName If the given name is already used by this owner, adjust the name to ensure uniqueness instead of returning an error.
		#' @param clusterID Cluster ID of a federated cluster where this object should be created.
		#' @param async If true, cluster permission will not be updated immediately, but instead at the next configured update interval.
		#' @return A Group object.
		project_create = function(group, select = NULL,
			ensureUniqueName = NULL, clusterID = NULL,
			async = NULL)
		{
			group <- c("group_class" = "project", group)
			self$groups_create(group = group, select = select, ensureUniqueName = ensureUniqueName, clusterID = clusterID, async = async)
		},

		#' @description A wrapper for `groups_update` that sets `group_class="project"`.
		#' @param group Group object.
		#' @param uuid The UUID of the Group to update.
		#' @param select An array of names of attributes to return in the response.
		#' @param async If true, cluster permission will not be updated immediately, but instead at the next configured update interval.
		#' @return A Group object.
		project_update = function(group, uuid, select = NULL, async = NULL)
		{
			group <- c("group_class" = "project", group)
			self$groups_update(group = group, uuid = uuid, select = select, async = async)
		},

		#' @description A wrapper for `groups_list` that adds a filter for `group_class="project"`.
		#' @param filters Filters to limit which objects are returned by their attributes.
		#' Refer to the [filters reference][] for more information about how to write filters.
		#' 
		#' [filters reference]: https://doc.arvados.org/api/methods.html#filters
		#' 
		#' @param where An object to limit which objects are returned by their attributes.
		#' The keys of this object are attribute names.
		#' Each value is either a single matching value or an array of matching values for that attribute.
		#' The `filters` parameter is more flexible and preferred.
		#' 
		#' @param order An array of strings to set the order in which matching objects are returned.
		#' Each string has the format `<ATTRIBUTE> <DIRECTION>`.
		#' `DIRECTION` can be `asc` or omitted for ascending, or `desc` for descending.
		#' 
		#' @param select An array of names of attributes to return from each matching object.
		#' @param distinct If this is true, and multiple objects have the same values
		#' for the attributes that you specify in the `select` parameter, then each unique
		#' set of values will only be returned once in the result set.
		#' 
		#' @param limit The maximum number of objects to return in the result.
		#' Note that the API may return fewer results than this if your request hits other
		#' limits set by the administrator.
		#' 
		#' @param offset Return matching objects starting from this index.
		#' Note that result indexes may change if objects are modified in between a series
		#' of list calls.
		#' 
		#' @param count A string to determine result counting behavior. Supported values are:
		#' 
		#'   * `"exact"`: The response will include an `items_available` field that
		#'     counts the number of objects that matched this search criteria,
		#'     including ones not included in `items`.
		#' 
		#'   * `"none"`: The response will not include an `items_avaliable`
		#'     field. This improves performance by returning a result as soon as enough
		#'     `items` have been loaded for this result.
		#' 
		#' 
		#' @param clusterID Cluster ID of a federated cluster to return objects from
		#' @param bypassFederation If true, do not return results from other clusters in the
		#' federation, only the cluster that received the request.
		#' You must be an administrator to use this flag.
		#' 
		#' @param includeTrash Include items whose `is_trashed` attribute is true.
		#' @return A GroupList object.
		project_list = function(filters = NULL, where = NULL,
			order = NULL, select = NULL, distinct = NULL,
			limit = NULL, offset = NULL, count = NULL,
			clusterID = NULL, bypassFederation = NULL,
			includeTrash = NULL)
		{
			filters[[length(filters) + 1]] <- list("group_class", "=", "project")
			self$groups_list(filters = filters, where = where, order = order, select = select, distinct = distinct, limit = limit, offset = offset, count = count, clusterID = clusterID, bypassFederation = bypassFederation, includeTrash = includeTrash)
		},

		#' @description An alias for `groups_delete`.
		#' @param uuid The UUID of the Group to delete.
		#' @return A Group object.
		project_delete = function(uuid)
		{
			self$groups_delete(uuid = uuid)
		},

		#' @description Test whether or not a project exists.
		#' @param uuid The UUID of the Group to return.
		#' @param select An array of names of attributes to return in the response.
		#' @param includeTrash Return group/project even if its `is_trashed` attribute is true.
		project_exist = function(uuid, select = NULL, includeTrash = NULL)
		{
			result <- try(self$groups_get(uuid = uuid, select = select, includeTrash = includeTrash))
			if(inherits(result, "try-error"))
				exists <- FALSE
			else
				exists <- result['group_class'] == "project"
			cat(format(exists))
		},

		#' @description A convenience wrapper for `project_update` to set project metadata properties.
		#' @param listProperties List of new properties.
		#' @param uuid UUID of the project to update.
		#' @return A Group object.
		project_properties_set = function(listProperties, uuid)
		{
			self$project_update(list("properties" = listProperties), uuid)
		},

		#' @description Get a project and update it with additional properties.
		#' @param properties List of new properties.
		#' @param uuid UUID of the project to update.
		#' @return A Group object.
		project_properties_append = function(properties, uuid)
		{
			proj <- private$get_project_by_list(uuid, list('uuid', 'properties'))
			newListOfProperties <- c(proj$properties, properties)
			uniqueProperties <- unique(unlist(newListOfProperties))
			newProperties <- suppressWarnings(newListOfProperties[which(newListOfProperties == uniqueProperties)])
			self$project_properties_set(newProperties, proj$uuid)
		},

		#' @description Get properties of a project.
		#' @param uuid The UUID of the project to query.
		project_properties_get = function(uuid)
		{
			private$get_project_by_list(uuid, list('uuid', 'properties'))$properties
		},

		#' @description Delete one property from a project by name.
		#' @param oneProp Name of the property to delete.
		#' @param uuid The UUID of the project to update.
		#' @return A Group object.
		project_properties_delete = function(oneProp, uuid)
		{
			projProp <- self$project_properties_get(uuid)
			projProp[[oneProp]] <- NULL
			self$project_properties_set(projProp, uuid)
		},

		#' @description Convenience wrapper of `links_list` to create a permission link.
		#' @param type The type of permission: one of `'can_read'`, `'can_write'`, or `'can_manage'`.
		#' @param uuid The UUID of the object to grant permission to.
		#' @param user The UUID of the user or group who receives this permission.
		#' @return A Link object if one was updated, else NULL.
		project_permission_give = function(type, uuid, user)
		{
			link <- list(
				'link_class' = 'permission',
				'name' = type,
				'head_uuid' = uuid,
				'tail_uuid' = user)
			self$links_create(link)
		},

		#' @description Find an existing permission link and update its level.
		#' @param typeOld The type of permission to find: one of `'can_read'`, `'can_write'`, or `'can_manage'`.
		#' @param typeNew The type of permission to set: one of `'can_read'`, `'can_write'`, or `'can_manage'`.
		#' @param uuid The UUID of the object to grant permission to.
		#' @param user The UUID of the user or group who receives this permission.
		#' @return A Link object if one was updated, else NULL.
		project_permission_update = function(typeOld, typeNew, uuid, user)
		{
			links <- self$links_list(filters = list(
					list('link_class', '=', 'permission'),
					list('name', '=', typeOld),
					list('head_uuid', '=', uuid),
					list('tail_uuid', '=', user)
				), select=list('uuid'), count = 'none')$items
			if (length(links) == 0) {
				cat(format('No permission granted'))
			} else {
				self$links_update(list('name' = typeNew), links[[1]]$uuid)
			}
		},

		#' @description Delete an existing permission link.
		#' @param type The type of permission to delete: one of `'can_read'`, `'can_write'`, or `'can_manage'`.
		#' @param uuid The UUID of the object to grant permission to.
		#' @param user The UUID of the user or group who receives this permission.
		#' @return A Link object if one was deleted, else NULL.
		project_permission_delete = function(type, uuid, user)
		{
			links <- self$links_list(filters = list(
					list('link_class', '=', 'permission'),
					list('name', '=', type),
					list('head_uuid', '=', uuid),
					list('tail_uuid', '=', user)
				), select=list('uuid'), count = 'none')$items
			if (length(links) == 0) {
				cat(format('No permission granted'))
			} else {
				self$links_delete(links[[1]]$uuid)
			}
		},

		#' @description Check for an existing permission link.
		#' @param type The type of permission to check: one of `'can_read'`, `'can_write'`, `'can_manage'`, or `NULL` (the default).
		#' @param uuid The UUID of the object to check permission on.
		#' @param user The UUID of the user or group to check permission for.
		#' @return If `type` is `NULL`, the list of matching permission links.
		#' Otherwise, prints and invisibly returns the level of the found permission link.
		project_permission_check = function(uuid, user, type = NULL)
		{
			filters <- list(
				list('link_class', '=', 'permission'),
				list('head_uuid', '=', uuid),
				list('tail_uuid', '=', user))
			if (!is.null(type)) {
				filters <- c(filters, list(list('name', '=', type)))
			}
			links <- self$links_list(filters = filters, count='none')$items
			if (is.null(type)) {
				links
			} else {
				print(links[[1]]$name)
			}
		},

		#' @description Return the host name of this client's Arvados API server.
		#' @return Hostname string.
		getHostName = function() private$host,

		#' @description Return the Arvados API token used by this client.
		#' @return API token string.
		getToken = function() private$token,

		#' @description Set the RESTService object used by this client.
		setRESTService = function(newREST) private$REST <- newREST,

		#' @description Return the RESTService object used by this client.
		#' @return RESTService object.
		getRESTService = function() private$REST
	),

	private = list(
		token = NULL,
		host = NULL,
		REST = NULL,
		numRetries = NULL,
		get_project_by_list = function(uuid, select = NULL)
		{
			self$groups_list(
				filters = list(list('uuid', '=', uuid), list('group_class', '=', 'project')),
				select = select,
				count = 'none'
			)$items[[1]]
		}
	),

	cloneable = FALSE
)
